<?php

require_once('CreditRequestDTO.php');
require_once('ResponseDTO.php');
require_once('AccessTokenRequestDTO.php');
require_once('TransaccionDTO.php');

/**
 * Class InstantCredit
 */
class ICredit
{
    //Entornos Diponibles
    const ENTORNO_PRODUCCION = 0;
    const ENTORNO_TEST = 1;
    const MODO_PEDIDO_ANTICIPADO_ON = 1;
    const MODO_PEDIDO_ANTICIPADO_OFF = 0;

    //Métodos del webservice
    const GET_ACCESS_TOKEN = 'GetAccessToken';
    const REFRESH_ACCESS_TOKEN = 'RefreshAccessToken';
    const SEND_CREDIT_REQUEST = 'SendCreditRequest';
    const SEND_REFUND_REQUEST = 'SendRefundRequest';
    const RECEIVE_CREDIT_CONFIRMATION = 'ReceiveCreditconfirmation';
    const SIMULADOR = 'Simulador';

    public static $icEstado = array('instantcredit_APROBADO', 'instantcredit_PENDIENTE', 'instantcredit_DECLINADO', 'instantcredit_TIMEOUT', 'instantcredit_CANCELLED_BY_END_USER');
    public static $textoEstado = array("Pago por %s: aprobado", "Pago por %s: Pendiente", "Pago por %s: cancelado", "Pago mediante %s cancelado por TimeOut", "Pago mediante %s cancelado por el Usuario");
    public static $colorEstado = array('#108510', '#4169E1', '#FA5656', '#FA5656', '#FA5656');
    public static $templateEstado = array('instantcredit_accepted', '', 'instantcredit_cancelled', 'order_cancelled', 'order_cancelled');
    public static $pagadoEstado = array(1, 0, 0, 0, 0);
    public static $emailEstado = array(true, false, false, false, false);

    const APROBADO = "instantcredit_APROBADO";
    const PENDIENTE = "instantcredit_PENDIENTE";
    const DECLINADO = "instantcredit_DECLINADO";
    const TIMEOUT = "instantcredit_TIMEOUT";
    const CANCELLED_BY_END_USER = "instantcredit_CANCELLED_BY_END_USER";


    static public $http_status_codes = array(
        100 => 'Informational: Continue',
        101 => 'Informational: Switching Protocols',
        102 => 'Informational: Processing',
        200 => 'Successful: OK',
        201 => 'Successful: Created',
        202 => 'Successful: Accepted',
        203 => 'Successful: Non-Authoritative Information',
        204 => 'Successful: No Content',
        205 => 'Successful: Reset Content',
        206 => 'Successful: Partial Content',
        207 => 'Successful: Multi-Status',
        208 => 'Successful: Already Reported',
        226 => 'Successful: IM Used',
        300 => 'Redirection: Multiple Choices',
        301 => 'Redirection: Moved Permanently',
        302 => 'Redirection: Found',
        303 => 'Redirection: See Other',
        304 => 'Redirection: Not Modified',
        305 => 'Redirection: Use Proxy',
        306 => 'Redirection: Switch Proxy',
        307 => 'Redirection: Temporary Redirect',
        308 => 'Redirection: Permanent Redirect',
        400 => 'Client Error: Bad Request',
        401 => 'Client Error: Unauthorized',
        402 => 'Client Error: Payment Required',
        403 => 'Client Error: Forbidden',
        404 => 'Client Error: Not Found',
        405 => 'Client Error: Method Not Allowed',
        406 => 'Client Error: Not Acceptable',
        407 => 'Client Error: Proxy Authentication Required',
        408 => 'Client Error: Request Timeout',
        409 => 'Client Error: Conflict',
        410 => 'Client Error: Gone',
        411 => 'Client Error: Length Required',
        412 => 'Client Error: Precondition Failed',
        413 => 'Client Error: Request Entity Too Large',
        414 => 'Client Error: Request-URI Too Long',
        415 => 'Client Error: Unsupported Media Type',
        416 => 'Client Error: Requested Range Not Satisfiable',
        417 => 'Client Error: Expectation Failed',
        418 => 'Client Error: I\'m a teapot',
        419 => 'Client Error: Authentication Timeout',
        420 => 'Client Error: Method Failure',
        422 => 'Client Error: Unprocessable Entity',
        423 => 'Client Error: Locked',
        424 => 'Client Error: Failed Dependency',
        425 => 'Client Error: Unordered Collection',
        426 => 'Client Error: Upgrade Required',
        428 => 'Client Error: Precondition Required',
        429 => 'Client Error: Too Many Requests',
        431 => 'Client Error: Request Header Fields Too Large',
        444 => 'Client Error: No Response',
        449 => 'Client Error: Retry With',
        450 => 'Client Error: Blocked by Windows Parental Controls',
        451 => 'Client Error: Unavailable For Legal Reasons',
        494 => 'Client Error: Request Header Too Large',
        495 => 'Client Error: Cert Error',
        496 => 'Client Error: No Cert',
        497 => 'Client Error: HTTP to HTTPS',
        499 => 'Client Error: Client Closed Request',
        500 => 'Server Error: Internal Server Error',
        501 => 'Server Error: Not Implemented',
        502 => 'Server Error: Bad Gateway',
        503 => 'Server Error: Service Unavailable',
        504 => 'Server Error: Gateway Timeout',
        505 => 'Server Error: HTTP Version Not Supported',
        506 => 'Server Error: Variant Also Negotiates',
        507 => 'Server Error: Insufficient Storage',
        508 => 'Server Error: Loop Detected',
        509 => 'Server Error: Bandwidth Limit Exceeded',
        510 => 'Server Error: Not Extended',
        511 => 'Server Error: Network Authentication Required',
        598 => 'Server Error: Network read timeout error',
        599 => 'Server Error: Network connect timeout error',
    );
    static private $urlTest = array(self::GET_ACCESS_TOKEN => 'https://test.instantcredit.net/api/oauth/token',
        self::REFRESH_ACCESS_TOKEN => 'https://test.instantcredit.net/api/oauth/refresh',
        self::SEND_CREDIT_REQUEST => 'https://test.instantcredit.net/api/credit/request',
        self::SEND_REFUND_REQUEST => 'https://test.instantcredit.net/api/refund/request',
        self::RECEIVE_CREDIT_CONFIRMATION => 'https://test.instantcredit.net/api/credit/confirmation',
        self::SIMULADOR => 'https://instantcredit.net/simulator/test/ic-simulator.js'
    );
    static private $urlProduccion = array(self::GET_ACCESS_TOKEN => 'https://api.instantcredit.net/api/oauth/token',
        self::REFRESH_ACCESS_TOKEN => 'https://api.instantcredit.net/api/oauth/refresh',
        self::SEND_CREDIT_REQUEST => 'https://api.instantcredit.net/api/credit/request',
        self::SEND_REFUND_REQUEST => 'https://api.instantcredit.net/api/refund/request',
        self::RECEIVE_CREDIT_CONFIRMATION => 'https://api.instantcredit.net/api/credit/confirmation',
        self::SIMULADOR => 'https://instantcredit.net/simulator/ic-simulator.js'
    );

    /** Funcion para obtener formateadas las cabeceras.
     * @param $headers
     * @return array
     */
    static public function ParseHeaders($headers)
    {
        $head = array();
        foreach ($headers as $k => $v) {
            $t = explode(':', $v, 2);
            if (isset($t[1])) {
                $head[trim($t[0])] = trim($t[1]);
            } else {
                $head[] = $v;
                if (preg_match("#HTTP/[0-9\.]+\s+([0-9]+)#", $v, $out)) {
                    $head['response_code'] = intval($out[1]);
                }
            }
        }
        return $head;
    }

    /**
     * @param $request
     * @param null $accessToken
     * @return null|resource
     */
    static private function GenerarPeticion($request, $accessToken = null)
    {
        if ($request != null) {
            $peticion = null;
            $header = "Content-Type: application/json\r\n";

            if (isset($accessToken) && !empty($accessToken)) {
                $header .= "Authorization: Bearer " . trim($accessToken) . "\r\n";
            }

            if ($request->requestType != null && $request->requestType != self::GET_ACCESS_TOKEN) {
                $header .= "Signature: " . $request->CalcularFirma() . "\r\n";
            }

            $peticionArray = array(
                'http' => $httpArray = array(
                    'method' => 'POST',
                    'header' => $header,
                    'content' => $request->toJson()
                )
            );
            PrestaShopLogger::addLog("SolicitudDeCredito " . date("d/m/y H:i:s") . 'Content: ' . $request->toJson());

            $peticion = stream_context_create($peticionArray);
        } 

        return $peticion;
    }

    /***
     * @param $peticion peticion a enviar
     * @param $entornoActivo integer ENTORNO_TEST o ENTORNO_PRODUCCION
     * @param $requestType string con el valor del método que queremos invocar en el webservice
     * @return ResponseDTO con el resultado parseado.
     */
    static private function EnviarPeticion($peticion, $entornoActivo, $requestType, $secretKey)
    {
        $response = file_get_contents(self::ObtenerURL($entornoActivo, $requestType), FALSE, $peticion);
        $cabeceras = self::ParseHeaders($http_response_header);
        return self::ProcesaRespuesta($response, $cabeceras, $secretKey);
    }

    static public function ProcesaRespuesta($response, $cabeceras, $secretKey)
    {
        $responseDto = new ResponseDTO();

        try {
            $responseDto->secretKey = $secretKey;
            $responseDto->signature = array_key_exists('Signature', $cabeceras) ? $cabeceras['Signature'] : null;
            $responseDto->httpResponseCode = array_key_exists('response_code', $cabeceras) ? $cabeceras['response_code'] : null;
            $responseDto->httpResponseDescri = $responseDto->httpResponseCode === TRUE ? self::$http_status_codes[$responseDto->httpResponseCode] : null;
            $responseDto->response = $response;
            $responseDto->responseOk = false;

            if ($responseDto->httpResponseCode != null) {
                $responseDto->responseOk = $responseDto->httpResponseCode == '200' ? true : false;
            } else {
                if ($response != null && $response != '') {
                    $responseDto->responseOk = true;
                }
            }

            if ($responseDto->responseOk == false && $responseDto->httpResponseCode != null) {
                $responseDto->errorMessage = $responseDto->httpResponseDescri;
            } elseif ($responseDto->responseOk == false && $responseDto->httpResponseCode == null) {
                $responseDto->errorMessage = 'No se ha obtenido respuesta';
            }

        } catch (Exception $e) {
            $responseDto->signature = null;
            $responseDto->httpResponseCode = null;
            $responseDto->httpResponseDescri = null;
            $responseDto->responseOk = false;
            $responseDto->response = null;
            $responseDto->errorMessage = $e->getMessage();
        }

        return $responseDto;
    }


    /**
     * Obtenemos la  URL dependiendo de si estamos en entorno Test o Producción
     * @param $entornoActivo integer ENTORNO_TEST o ENTORNO_PRODUCCION
     * @param $metodo string con el valor del método que queremos invocar en el webservice
     * @return string con la url necesaria para la llamada al método.
     */
    static public function ObtenerURL($entornoActivo, $metodo)
    {

        $arrayEnUso = null;

        switch ($entornoActivo) {
            case self::ENTORNO_TEST:
                $arrayEnUso = self::$urlTest;
                break;
            case self::ENTORNO_PRODUCCION:
                $arrayEnUso = self::$urlProduccion;
                break;
            default:
                $arrayEnUso = self::$urlTest;
                break;

        }

        if ($arrayEnUso != null && is_array($arrayEnUso) && count($arrayEnUso) > 0) {
            return $arrayEnUso[$metodo];
        }

        return null;

    }

    /**
     * @param $creditRequest CreditRequestDTO con los datos de la petición
     * @param $accessToken string de seguridad
     * @param $entornoActivo integer ENTORNO_TEST o ENTORNO_PRODUCCION
     * @return ResponseDTO con el resultado.
     */
    static function SendCreditRequest($creditRequest, $accessToken, $entornoActivo)
    {
        $peticion = self::GenerarPeticion($creditRequest, $accessToken);    
        $responseDTO = self::EnviarPeticion($peticion, $entornoActivo, $creditRequest->requestType, $creditRequest->secretKey);
        return $responseDTO;
    }

    /**
     * @param $creditConfirmationResponse CreditConfirmationResponseDTO con los datos de la petición
     * @param $accessToken string de seguridad
     * @param $entornoActivo integer ENTORNO_TEST o ENTORNO_PRODUCCION
     * @return ResponseDTO con el resultado.
     */
    static function SendCreditConfirmationResponse($creditConfirmationResponse, $accessToken, $entornoActivo)
    {
        $peticion = self::GenerarPeticion($creditConfirmationResponse, $accessToken);
        $responseDTO = self::EnviarPeticion($peticion, $entornoActivo, $creditConfirmationResponse->requestType, $creditConfirmationResponse->secretKey);
        return $responseDTO;
    }

    /**
     * Método para obtener el Token
     * @param $entornoActivo integer ENTORNO_TEST o ENTORNO_PRODUCCION
     * @param $clientId string con el identificador del cliente
     * @param $userName string con el nombre de usuario
     * @param $password string con el password
     * @return ResponseDTO con el resultado.
     */
    static function GetAccessToken($clientId, $userName, $password, $entornoActivo)
    {
		

            $accesTokenRequest = new AccessTokenRequestDTO();
            $accesTokenRequest->clientId = $clientId;
            $accesTokenRequest->userName = $userName;
            $accesTokenRequest->password = $password;
            $peticion = self::GenerarPeticion($accesTokenRequest);
            $responseDTO = self::EnviarPeticion($peticion, $entornoActivo, $accesTokenRequest->requestType, null);

        
        
        return $responseDTO;
    }
}